using System;
using System.Runtime.Serialization;

using Team_Project.PersistencyManagers.Protocols;

namespace Team_Project.Exceptions
{
	/// <summary>
	/// Classe base per le eccezioni di Team project
	/// </summary>
	[Serializable]
	public class TeamProjectException : Exception
	{
		/// <summary>
		/// Costruttore per serializzazione
		/// </summary>
		/// <param name="info"></param>
		/// <param name="context"></param>
		protected TeamProjectException(SerializationInfo info, StreamingContext context):
		base(info,context){}

		/// <summary>
		/// Costruisce un'istanza
		/// </summary>
		public TeamProjectException():base()
		{}

		/// <summary>
		/// Costruisce un'istanza
		/// </summary>
		/// <param name="message">Messaggio d'errore</param>
		public TeamProjectException(string message):base(message)
		{}

		/// <summary>
		/// Costruisce un'istanza
		/// </summary>
		/// <param name="message">Messaggio d'errore</param>
		/// <param name="inner">Eccezione interna</param>
		public TeamProjectException(string message, Exception inner):base(message,inner)
		{}
	}

	/// <summary>
	/// Eccezione generata se si tenta di utilizzare un protocollo per store o per
	/// read quando questi non sono supportati
	/// </summary>
	[Serializable]
	public class DirectionNotSupportedException : TeamProjectException, ISerializable
	{
		/// <summary>
		/// Protocollo che ha generato l'eccezione
		/// </summary>
		protected IPersistencyProtocol protocol;
		/// <summary>
		/// Direzione richiesta
		/// </summary>
		protected Direction direction;

		/// <summary>
		/// Costruttore per serializzazione
		/// </summary>
		/// <param name="info"></param>
		/// <param name="context"></param>
		protected DirectionNotSupportedException(SerializationInfo info, StreamingContext context):
		base(info,context)
		{
			Type pt = (Type)info.GetValue("ProtocolType",typeof(System.Type));
			protocol = (IPersistencyProtocol)info.GetValue("Protocol",pt);
		}

		/// <summary>
		/// Costruisce un'istanza
		/// </summary>
		public DirectionNotSupportedException():
			base()
		{}
		
		/// <summary>
		/// Costruisce un'istanza
		/// </summary>
		/// <param name="protocol">Protocollo che ha generato l'eccezione</param>
		/// <param name="dir">Direzione richiesta</param>
		public DirectionNotSupportedException(IPersistencyProtocol protocol, Direction dir):
			this(protocol,dir,protocol.Name + " does not support " + dir.ToString())
		{}

		/// <summary>
		/// Costruisce un'istanza
		/// </summary>
		/// <param name="protocol">Protocollo che ha generato l'eccezione</param>
		/// <param name="dir">Direzione richiesta</param>
		/// <param name="message">Messaggio d'errore ulteriore</param>
		public DirectionNotSupportedException(IPersistencyProtocol protocol, Direction dir, string message):
			this(protocol,dir,message,null)
		{}

		/// <summary>
		/// Costruisce un'istanza
		/// </summary>
		/// <param name="protocol">Protocollo che ha generato l'eccezione</param>
		/// <param name="dir">Direzione richiesta</param>
		/// <param name="message">Messaggio d'errore ulteriore</param>
		/// <param name="inner">Eccezione interna</param>
		public DirectionNotSupportedException(IPersistencyProtocol protocol, Direction dir, string message, Exception inner)
			:base(message,inner)
		{
			this.protocol = protocol;
			this.direction = dir;
		}

		/// <summary>
		/// Ottiene un riferimento al protocollo che ha generato l'eccezzione
		/// </summary>
		public IPersistencyProtocol Protocol
		{
			get{return protocol;}
		}

		/// <summary>
		/// Ottiene la direzione richiesta
		/// </summary>
		public Direction Direction
		{
			get{return direction;}
		}
		#region ISerializable Members

		/// <summary>
		/// Funzione per la serializzazione
		/// </summary>
		/// <param name="info"></param>
		/// <param name="context"></param>
		public override void GetObjectData(SerializationInfo info, StreamingContext context)
		{
			base.GetObjectData(info,context);
			info.AddValue("ProtocolType",protocol.GetType());
			info.AddValue("Protocol",protocol);
		}

		#endregion
	}
}
